#!/usr/bin/env bash
#
# entropy_check.sh – Run `ent` + `rngtest` on a binary keystream
#
# Usage:
#   ./entropy_check.sh <binary_keystream>
#
#   <binary_keystream> – path to the file produced by build_pad.sh
#                         (or any raw binary key you want to test)
#
# The script reports:
#   * Entropy (bits per byte) – should be ≈ 8.0
#   * Percentage of ones – should be ~50%
#   * chi‑square p‑value – must be > 0.01 to be considered OK
#   * rngtest p‑value – also must be > 0.01
#
# If either test fails, the script exits with status 1; otherwise it exits 0.
# This makes it easy to use in CI pipelines or larger scripts.

set -euo pipefail

# ----------------------------------------------------------------------
# Helper: print usage and exit
# ----------------------------------------------------------------------
usage() {
    echo "Usage: $0 <binary_keystream>"
    exit 1
}
[[ $# -eq 1 ]] || usage

STREAM_FILE="$1"

# ----------------------------------------------------------------------
# Verify that required tools are installed
# ----------------------------------------------------------------------
if ! command -v ent >/dev/null 2>&1; then
    echo "ERROR: 'ent' is not installed. Install with:" >&2
    echo "       sudo apt install ent" >&2
    exit 1
fi

if ! command -v rngtest >/dev/null 2>&1; then
    echo "ERROR: 'rngtest' is not installed. Install with:" >&2
    echo "       sudo apt install rng-tools" >&2
    exit 1
fi

# ----------------------------------------------------------------------
# Basic sanity checks on the input file
# ----------------------------------------------------------------------
if [[ ! -f "$STREAM_FILE" ]]; then
    echo "ERROR: File '$STREAM_FILE' does not exist." >&2
    exit 1
fi

if [[ ! -r "$STREAM_FILE" ]]; then
    echo "ERROR: Cannot read file '$STREAM_FILE' (permission denied)." >&2
    exit 1
fi

# ----------------------------------------------------------------------
# Run ent – capture its human‑readable output
# ----------------------------------------------------------------------
ENT_OUTPUT=$(ent "$STREAM_FILE")
# Extract the fields we care about
ENTROPY=$(echo "$ENT_OUTPUT" | awk '/Entropy/ {print $2}')
ONES_PCT=$(echo "$ENT_OUTPUT" | awk '/Optimum/ {print $4}' | tr -d '%')
CHISQ_PVAL=$(echo "$ENT_OUTPUT" | awk '/Chi/ {print $5}')

# ----------------------------------------------------------------------
# Run rngtest – it prints a single line with a p‑value at the end
# ----------------------------------------------------------------------
RNGTEST_OUTPUT=$(rngtest < "$STREAM_FILE")
# Example line: "rngtest: 0.667 (passed 1/1 tests)"
RNG_PVAL=$(echo "$RNGTEST_OUTPUT" | awk -F'[()]' '{print $2}' | awk '{print $1}')

# ----------------------------------------------------------------------
# Summarise results
# ----------------------------------------------------------------------
echo "=== ENT results ==="
printf "Entropy          : %.3f bits/byte (ideal = 8.0)\n" "$ENTROPY"
printf "Ones percentage : %.2f%% (ideal ≈ 50%%)\n" "$ONES_PCT"
printf "Chi‑square p‑val: %s\n" "$CHISQ_PVAL"

echo "=== RNGTEST results ==="
printf "p‑value (rngtest): %s\n" "$RNG_PVAL"

# ----------------------------------------------------------------------
# Determine pass/fail (both p‑values must be > 0.01)
# ----------------------------------------------------------------------
PASS=true

# Helper to compare floating‑point numbers (bc returns 1 for true)
float_gt() { bc -l <<< "$1 > $2" ; }

if ! float_gt "$ENTROPY" "7.9"; then
    echo "FAIL: Entropy is too low (< 7.9 bits/byte)." >&2
    PASS=false
fi

if ! float_gt "$CHISQ_PVAL" "0.01"; then
    echo "FAIL: chi‑square p‑value ≤ 0.01 (non‑random)." >&2
    PASS=false
fi

if ! float_gt "$RNG_PVAL" "0.01"; then
    echo "FAIL: rngtest p‑value ≤ 0.01 (non‑random)." >&2
    PASS=false
fi

if $PASS; then
    echo "=== RESULT: PASS – keystream looks random enough for OTP use ==="
    exit 0
else
    echo "=== RESULT: FAIL – please generate more entropy and retry ===" >&2
    exit 1
fi